import React, { useEffect, useState } from 'react';
import { db, auth, googleProvider } from './firebase';
import { newDeck, dealHands, legalMovesForCard, applyMoveToState } from './game/engine';
import { collection, doc, setDoc, getDoc, onSnapshot, updateDoc } from 'firebase/firestore';
import { signInAnonymously, signInWithPopup } from 'firebase/auth';

const DEFAULT_PLAYERS = 4;

function makePlayers(n) {
  return Array.from({length:n}, (_,i) => ({
    id: `p${i+1}`,
    name: `Player ${i+1}`,
    isBot: false,
    pegs: Array.from({length:4}, ()=>-1),
    finished: 0,
    hand: []
  }));
}

export default function App(){
  const [numPlayers, setNumPlayers] = useState(DEFAULT_PLAYERS);
  const [botMask, setBotMask] = useState(0);
  const [gameDocId, setGameDocId] = useState(null);
  const [game, setGame] = useState(null);
  const [log, setLog] = useState([]);
  const [user, setUser] = useState(null);

  useEffect(()=>{ // auth
    signInAnonymously(auth).then(res=> setUser(res.user)).catch(()=>{});
  },[]);

  const pushLog = (s) => setLog(l => [s, ...l].slice(0,200));

  const createGame = async () => {
    const deck = newDeck();
    const hands = dealHands(deck, numPlayers);
    const players = makePlayers(numPlayers).map((p,i)=> ({ ...p, isBot: !!(botMask & (1<<i)), hand: hands[i] || [] }));
    const initial = { players, deck, turn: 0, history: [], status: 'playing' };
    const gamesRef = collection(db, 'games');
    const id = 'game-' + Date.now();
    await setDoc(doc(gamesRef, id), initial);
    setGameDocId(id);
    pushLog('Created game ' + id);
    // subscribe
    const unsub = onSnapshot(doc(gamesRef, id), snap => {
      setGame(snap.data());
    });
  };

  const playMove = async (playerIndex, cardIndex, pegIdx) => {
    if (!game) return;
    try {
      const newState = applyMoveToState(game, playerIndex, cardIndex, pegIdx);
      // write authoritative state to Firestore
      await setDoc(doc(db, 'games', gameDocId), newState);
      pushLog(`Player ${playerIndex+1} moved peg ${pegIdx}`);
    } catch (e) {
      pushLog('Invalid move: ' + e.message);
    }
  };

  // simplistic bot runner: if it's a bot's turn, client will play for it
  useEffect(()=>{
    if (!game) return;
    const t = game.turn;
    const p = game.players[t];
    if (!p) return;
    if (!p.isBot) return;
    // find first playable
    setTimeout(async ()=>{
      for (let ci=0;ci<p.hand.length;ci++){
        const moves = legalMovesForCard(p, p.hand[ci]);
        if (moves.length>0){
          await playMove(t, ci, moves[0].pegIdx);
          break;
        }
      }
    }, 600);
  }, [game]);

  return (
    <div style={{padding:20,fontFamily:'sans-serif'}}>
      <h1>Pegs & Jokers — Firebase Starter</h1>
      <div style={{marginTop:10}}>
        Players:
        <select value={numPlayers} onChange={e=>setNumPlayers(Number(e.target.value))} style={{marginLeft:8}}>
          {[2,3,4,5,6].map(n=> <option key={n} value={n}>{n}</option>)}
        </select>
        <label style={{marginLeft:16}}>Bots mask:
          <input value={botMask} onChange={e=>setBotMask(Number(e.target.value)||0)} style={{marginLeft:8,width:70}}/>
        </label>
        <button onClick={createGame} style={{marginLeft:12}}>Create Game</button>
      </div>

      <div style={{display:'flex',gap:20,marginTop:20}}>
        <div style={{flex:1}}>
          <div style={{border:'1px solid #ccc',padding:10}}>
            <div style={{height:360, backgroundImage:'url(/board.jpg)', backgroundSize:'contain', backgroundRepeat:'no-repeat', backgroundPosition:'center'}} />
            <div style={{marginTop:8,fontSize:12,color:'#666'}}>Replace public/board.jpg with your provided photo.</div>
          </div>
          <div style={{marginTop:10}}>
            {game ? game.players.map((pl,idx)=> (
              <div key={idx} style={{border:'1px solid #eee',padding:8,marginBottom:6}}>
                <strong>{pl.name} {pl.isBot?'(Bot)':''}</strong>
                <div>Finished: {pl.finished}</div>
                <div style={{marginTop:6}}>Pegs: {pl.pegs.map((p,i)=> <span key={i} style={{marginRight:6}}>{i}:{p}</span>)}</div>
                <div style={{marginTop:6}}>Hand: {pl.hand.join(', ')}</div>
                {game.turn===idx && !pl.isBot && (
                  <div style={{marginTop:6,color:'#070'}}>Your turn</div>
                )}
              </div>
            )) : <div>No game</div>}
          </div>
        </div>

        <div style={{width:360}}>
          <div style={{border:'1px solid #ddd',padding:10}}>
            <div style={{fontWeight:600}}>Game Controls</div>
            <div style={{marginTop:8}}>Game ID: {gameDocId || '—'}</div>
            <div style={{marginTop:8}}>Turn: {game ? ('Player ' + (game.turn+1)) : '—'}</div>
            <div style={{marginTop:12}}>
              <button onClick={()=>{ /* sign in with Google */ signInWithPopup(auth, googleProvider).then(()=>{}).catch(()=>{});}}>Sign in with Google</button>
            </div>
          </div>

          <div style={{marginTop:12,border:'1px solid #eee',padding:10}}>
            <div style={{fontWeight:600}}>Action Log</div>
            <div style={{height:300,overflow:'auto',marginTop:6}}>
              {log.map((l,i)=> <div key={i} style={{fontSize:12}}>{l}</div>)}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
