

# Purpose: Provide UDFs to support book's Vet's office scenario scripts.
# Import modules with accessors so that UDFs will recognize them.

import datetime as dt
import pandas as pd
import numpy as np
import openpyxl as op
import os
import shutil
import sys
import win32com


# UDF Definition. 
def auto_email(Subject, Body_Text, HTML_Object, Emails):
    # Purpose: Send email with simply body text and one HTML table to one or more email addresses.
    # Subject: The Subject line text
    # Body_Text: Text string to appear at top of email body.
    # HTML_Object: HTML to appear as table at bottom of email body. Enter '' if no such table desired.
    # Emails: One or more email addresses. Strings. Delineate multiples with semi-colons.
    
    import win32com.client
    outlook_obj = win32com.client.Dispatch('Outlook.Application')
    email_obj          = outlook_obj.CreateItem(0)
    email_obj.Subject  = Subject
    email_obj.HTMLBody = '<html><body>' + Body_Text + '<br><br>' + HTML_Object + '</body></html>'
    email_obj.To       = Emails
    email_obj.Send()
    # Status message.
    print('Email sent: ' + Subject + ' to ' + Emails + '.\n')

    # No returned variable.   
    # End function definition.


# UDF Definition.
def custom_vet_export(Today_obj, dfClients, dfApps, dfArchive):
    # Purpose: Export dataframes to three tabs in Excel from book case study.
    File     = Today_obj.strftime('%Y %m%d ' + '- Vet.xlsx')
    Folder   = 'C:/Customers/'
    Pathname = Folder + File
    
    # Create writer object to receive dataframes.
    wr_obj = pd.ExcelWriter(Pathname, 
                            engine  = 'openpyxl')
    
    # Define dictionaries.
    dict_A = {'df' : dfClients, 'Tab' : 'Clients'}
    dict_B = {'df' : dfApps,    'Tab' : 'Appointments'}
    dict_C = {'df' : dfArchive, 'Tab' : 'Archive'}
    dict_list = [dict_A, dict_B, dict_C]
    
    # Populate writer object with each dataframe in dictionary list.
    for dict in dict_list:
        dict['df'].to_excel(wr_obj, 
                           sheet_name = dict['Tab'], 
                           index = False, 
                           freeze_panes = (1, 2))    
    # Close writer object.
    wr_obj.close()

    # Create workbook object to format Excel file.
    workbook_obj = op.load_workbook(Pathname)
    
    # Apply standardized fofmatting to all three tabs.
    sheet_list = [dict_A['Tab'], dict_B['Tab'], dict_C['Tab']]
    for sheet in sheet_list:
        sheet_working = workbook_obj[sheet]
        # Apply filter to cell range.
        sheet_working.auto_filter.ref= 'A1:P1'
        # Override Excel vertical alignment.
        for row in sheet_working.iter_rows():
            for cell in row:
                cell.alignment = op.styles.Alignment(vertical='top')
        # Format top row.
        for cell in sheet_working[1:1]:
            cell.alignment = op.styles.Alignment(wrap_text  = True,
                                                 vertical   = 'bottom',
                                                 horizontal = 'center')
        # Format select columns.
        for cell in sheet_working['E']:
            cell.alignment = op.styles.Alignment(wrap_text = True,
                                                 vertical   = 'top')
        for col in ['F', 'G']:
            for cell in sheet_working[col]:
                cell.number_format = '0.0'
        # Customize column widths.
        sheet_working.column_dimensions['A'].width = 20
        sheet_working.column_dimensions['B'].width = 15
        sheet_working.column_dimensions['E'].width = 25
    
    # Save and close both workbook object and Excel file.
    workbook_obj.save(Pathname)
    workbook_obj.close()
    
    # Status message.
    print('Excel file saved and formatted:')
    print(Pathname + '\n')
    
    # No value(s) returned.
    # End function definition.


# UDF Definition.
def custom_vet_import(Date_obj):
    # Purpose: Load three tabs from Vet file for book example. 
    # Note: Pass Date_obj as datetime object.
    
    # Define file Pathname    
    File     = Date_obj.strftime('%Y %m%d') + ' - Vet.xlsx'
    Folder   = 'C:/Customers/'
    Pathname = Folder + File
    
    # Define dictionaries.
    dict_A = {'Tab' : 'Clients'}
    dict_B = {'Tab' : 'Appointments'}
    dict_C = {'Tab' : 'Archive'}
    dict_list = [dict_A, dict_B, dict_C]
    
    # Load tabs into dataframes.
    for dict in dict_list:
        dict['df'] = pd.read_excel(Pathname, 
                                  sheet_name = dict['Tab']) 

    # Status message.
    print('Excel File Import: Completed.')
    print(' - Pathname: ' + Pathname + '\n')

    # Return three dataframes.
    return dict_A['df'], dict_B['df'], dict_C['df'] 
    # End function definition.


# UDF Definition.
def pause():
    # Purpose: Pause execution.
    input('Hit ENTER to continue >')
    print('')
    # No value(s) returned.
    # End function definition.


# UDF Definition.
def print_header(Title):
   # Purpose: Print header with line underneath.
    print(Title)
    Line  = '-' * len(Title)
    print(Line)
    # No value(s) returned.
    # End function definition.
    

# UDF Definition.
def set_date(Type_of_Date):    
    # Purpose: Create datetime object for desired report or other date.
    # Notes: The Type of Date paramater represents the kind of date to be used
    #        as a prompt (e.g. 'Please set Business Date:')
    print('Please set ' + Type_of_Date + ':')
    Now_obj   = dt.datetime.now()
    Today_obj = dt.datetime(Now_obj.year, Now_obj.month, Now_obj.day)
    print(' - Today Date is ' + Today_obj.strftime('%A, %Y-%m-%d') + '.')
    Answer = input(' - Enter: 1 to use Today Date;\n' +
                   '          2 for 03/19/26;\n'      + 
                   '          3 for 03/20/26; or,\n'  +
                   '          99 for custom date>')
    if Answer == '1':
        Date_obj = Today_obj
    elif Answer == '2':
        Date_obj = dt.datetime(2026, 3, 19)      
    elif Answer == '3':
        Date_obj = dt.datetime(2026, 3, 20)      
    else:
        Date_String = input('\n - Enter date in MM/DD/YYYY format>')
        Date_obj = pd.to_datetime(Date_String)
    # Create string version.
    Date_str = Date_obj.strftime('%m/%d/%Y')
    
    # Status message.
    print('\n' + Type_of_Date + ' set to: ' + Date_str + '\n')
    
    return Date_obj, Date_str
    # End function definition.    


# UDF Definition.
def standard_display_options():
    # Purpose: Load standard display options.
    pd.reset_option('all') 
    pd.set_option('display.max_columns', None)
    pd.set_option('display.width', 55)
    pd.set_option('display.max_rows', None)
    pd.set_option('display.precision', 2)
    # Status message.
    print('Display settings set.')
    # No variable is returned.
    # End function definition.
