package org.mi_plus_plus;

import java.io.IOException;
import java.nio.file.*;
import java.util.*;

/**
 * MiPlusPlus - full interpreter for .mic files
 * Supports:
 *  - output:
 *  - import <package>
 *  - load <package>
 *  - add <var>, set <var> to <value>, show <var>
 *  - when <var> was <value> run:  (alternative if)
 *
 * Packages:
 *  - console
 *  - apps
 */
public class MiPlusPlus {

    // ====== STATE ======
    private static final Set<String> importedPackages = new HashSet<>();
    private static final Set<String> activePackages = new HashSet<>();
    private static final Map<String, Integer> variables = new HashMap<>();

    // ====== MAIN ======
    public static void main(String[] args) {
        try {
            List<Path> micFiles = findMicFiles();
            if (micFiles.isEmpty()) {
                System.out.println("⚠️ No .mic files found in project.");
                return;
            }

            showFileMenu(micFiles);
            List<Path> selected = askUserChoice(micFiles);
            if (selected.isEmpty()) {
                System.out.println("ℹ️ Nothing selected. Exiting.");
                return;
            }

            for (Path file : selected) {
                runFile(file);
            }

        } catch (IOException e) {
            System.err.println("❌ Error scanning project: " + e.getMessage());
        }
    }

    // ====== FILE SEARCH ======
    private static List<Path> findMicFiles() throws IOException {
        List<Path> micFiles = new ArrayList<>();
        Files.walk(Paths.get("."))
                .filter(p -> p.toString().endsWith(".mic"))
                .forEach(micFiles::add);
        return micFiles;
    }

    private static void showFileMenu(List<Path> micFiles) {
        System.out.println("🔎 Found .mic files:");
        for (int i = 0; i < micFiles.size(); i++) {
            System.out.println((i + 1) + ") " + micFiles.get(i));
        }
        System.out.println("Enter file numbers separated by space (e.g., 1 2), or 0 to exit:");
    }

    private static List<Path> askUserChoice(List<Path> micFiles) {
        Scanner scanner = new Scanner(System.in);
        String input = scanner.nextLine().trim();
        if (input.equals("0")) return Collections.emptyList();

        List<Path> chosen = new ArrayList<>();
        String[] parts = input.split("\\s+");
        for (String part : parts) {
            try {
                int idx = Integer.parseInt(part) - 1;
                if (idx >= 0 && idx < micFiles.size()) {
                    chosen.add(micFiles.get(idx));
                } else {
                    System.err.println("⚠️ Invalid index: " + part);
                }
            } catch (NumberFormatException e) {
                System.err.println("⚠️ Not a number: " + part);
            }
        }
        return chosen;
    }

    // ====== RUN FILE ======
    private static void runFile(Path file) {
        System.out.println("\n=== Running " + file + " ===");
        try {
            String code = Files.readString(file);
            runProgram(code, file.toString());
        } catch (IOException e) {
            System.err.println("❌ Failed to read file " + file + ": " + e.getMessage());
        }
    }

    // ====== PROGRAM EXECUTION ======
    private static void runProgram(String code, String fileName) {
        String[] lines = code.split("\\r?\\n");
        int lineNum = 0;

        while (lineNum < lines.length) {
            String rawLine = lines[lineNum];
            lineNum++;
            String line = rawLine.trim();
            if (line.isEmpty() || line.startsWith("#")) continue;

            // handle potential multi-line blocks for 'when'
            if (line.startsWith("when ")) {
                lineNum = handleWhenBlock(lines, lineNum - 1, line, fileName, lineNum);
            } else {
                executeLine(line, fileName, lineNum);
            }
        }
    }

    // ====== HANDLE LINE ======
    private static void executeLine(String line, String fileName, int lineNum) {
        if (line.startsWith("output:")) {
            handleOutput(line);
        } else if (line.startsWith("import ")) {
            String pkg = line.substring(7).trim();
            importPackage(pkg);
        } else if (line.startsWith("load ")) {
            String pkg = line.substring(5).trim();
            loadPackage(pkg, fileName, lineNum);
        } else if (line.startsWith("add ")) {
            String var = line.substring(4).trim();
            variables.put(var, 0);
            System.out.println("[Var] Added " + var + " = 0");
        } else if (line.startsWith("set ")) {
            String[] parts = line.substring(4).split(" to ");
            if (parts.length == 2) {
                String var = parts[0].trim();
                try {
                    int value = Integer.parseInt(parts[1].trim());
                    variables.put(var, value);
                    System.out.println("[Var] Set " + var + " = " + value);
                } catch (NumberFormatException e) {
                    System.err.println("[Error] Invalid number in set: " + line);
                }
            } else {
                System.err.println("[Error] Invalid set syntax: " + line);
            }
        } else if (line.startsWith("show ")) {
            String var = line.substring(5).trim();
            if (variables.containsKey(var)) {
                System.out.println(variables.get(var));
            } else {
                System.err.println("[Error] Variable '" + var + "' not defined");
            }
        } else {
            System.err.println("[Error] File: " + fileName + ", Line " + lineNum +
                    ": Unknown command -> " + line);
        }
    }

    // ====== HANDLE WHEN BLOCK ======
    private static int handleWhenBlock(String[] lines, int currentLineIndex, String line, String fileName, int lineNum) {
        // when x was 5 run:
        try {
            String header = line.substring(5).trim(); // x was 5 run:
            if (!header.endsWith("run:")) {
                System.err.println("[Error] Invalid when syntax: " + line);
                return currentLineIndex + 1;
            }
            header = header.substring(0, header.length() - 4).trim(); // x was 5
            String[] parts = header.split(" was ");
            if (parts.length != 2) {
                System.err.println("[Error] Invalid when syntax: " + line);
                return currentLineIndex + 1;
            }
            String var = parts[0].trim();
            int value = Integer.parseInt(parts[1].trim());

            // check variable
            if (variables.getOrDefault(var, Integer.MIN_VALUE) == value) {
                // run block
                int blockLine = currentLineIndex + 1;
                while (blockLine < lines.length) {
                    String blockRaw = lines[blockLine];
                    if (!blockRaw.startsWith("\t")) break; // end of block
                    String blockLineTrimmed = blockRaw.substring(1);
                    executeLine(blockLineTrimmed, fileName, blockLine + 1);
                    blockLine++;
                }
            }

            // return next line index
            int blockLine = currentLineIndex + 1;
            while (blockLine < lines.length) {
                if (!lines[blockLine].startsWith("\t")) break;
                blockLine++;
            }
            return blockLine;

        } catch (Exception e) {
            System.err.println("[Error] Invalid when block at line " + lineNum + ": " + e.getMessage());
            return currentLineIndex + 1;
        }
    }

    // ====== COMMAND HANDLERS ======
    private static void handleOutput(String line) {
        String text = line.substring(7).trim();
        System.out.println(text);
    }

    private static void importPackage(String pkg) {
        if (!importedPackages.contains(pkg)) {
            System.out.println("📦 Package '" + pkg + "' imported for the first time.");
            importedPackages.add(pkg);
        } else {
            System.out.println("📦 Package '" + pkg + "' already imported.");
        }
    }

    private static void loadPackage(String pkg, String fileName, int lineNum) {
        if (!importedPackages.contains(pkg)) {
            System.err.println("[Error] File: " + fileName + ", Line " + lineNum +
                    ": Package '" + pkg + "' must be imported before load.");
            return;
        }

        if (!activePackages.contains(pkg)) {
            System.out.println("✅ Package '" + pkg + "' activated.");
            activePackages.add(pkg);

            // emulate package work
            if (pkg.equals("console")) {
                System.out.println("[Console] Console is ready");
            } else if (pkg.equals("apps")) {
                System.out.println("[Apps] Apps system loaded");
            }

        } else {
            System.out.println("✅ Package '" + pkg + "' already active.");
        }
    }
}
