// KARL'S ULTIMATE DIY DYNO — V1.3 (Brake + Inertia Dyno + EGT + NAU7802)  — Improved 2026-Feb-16 
// Now using LittleFS for HTML files (index.html and settings.html)
// MCP9600 multi-sensor support front page and report
// Load cell: switched from HX711 to Adafruit NAU7802

#include <WiFi.h>
#include <ESPAsyncWebServer.h>
#include <AsyncTCP.h>
#include <DNSServer.h>
#include <LittleFS.h>
#include <deque>
#include <vector>
#include <Preferences.h>
#include <ESPmDNS.h>
#include <freertos/FreeRTOS.h>
#include <freertos/task.h>
#include <sstream>
#include <cmath>

#include <Wire.h>
#include <Adafruit_MCP9600.h>
#include <Adafruit_NAU7802.h>  // ADDED: Adafruit NAU7802 library

// ─── MCP9600 addresses scan ──────────────────────────────────────
const uint8_t MCP_ADDRS[8] = {0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67};
Adafruit_MCP9600 mcps[8];
bool mcp_oks[8] = {false, false, false, false, false, false, false, false};
uint8_t active_mcps = 0;

// ─── EGT tracking for PDF ─────────────────────────────────────────────────
float peakEGT = 0.0f;
float finalEGT = 0.0f;
float runAmbient = 0.0f;

const char* ap_ssid = "karlsdyno";
const char* ap_password = "karlsdyno";

AsyncWebServer server(80);
AsyncWebSocket ws("/ws");
DNSServer dnsServer;

// pins for esp32-s3 devboard
const int HALL_PIN      = 16;
const int DRUM_PIN      = 15;

Adafruit_NAU7802 nau;  // ADDED: NAU7802 instance

std::deque<float> rpmBuffer;
std::deque<float> drumRpmBuffer;
const int BUFFER_SIZE = 32;
float RPM_FILTER_ALPHA = 0.50f;

float CALIBRATION_FACTOR = 3993.567890;  // Will need full recalibration!
float LEVER_ARM_METERS = 0.25;
float MAX_RPM = 16000;
float MAX_HP = 15;
float MAX_TORQUE = 15;
float PULSES_PER_REVOLUTION = 1.0f;
float PULSES_PER_REV_DRUM = 1.0f;
float DRUM_MASS = 10.0f;
float DRUM_DIAM_METERS = 0.3f;

String DYNO_MODE = "brake";
String DRUM_INERTIA_TYPE = "solid";
float DRUM_WALL_THICKNESS_CM = 2.0f;
float DRUM_INERTIA_CUSTOM = 0.5f;
String RPM_SOURCE = "spark";
float VIRTUAL_GEAR_RATIO = 1.0f;

// Inertia-mode tuning constants
const int   MIN_HIST_POINTS_INERTIA   = 6;
const float MIN_DRUM_RPM_INERTIA      = 15.0f;
const float MIN_ACCEL_RPM_PER_S       = 3.0f;
const float SLOPE_FILTER_NEW_ALPHA    = 0.80f;

struct LiveData {
  float rpm = 0;
  float hp = 0;
  float torque = 0;
  float peakHP = 0;
  float peakHP_RPM = 0;
  float peakTorque = 0;
  float peakTorque_RPM = 0;
};
LiveData current;

volatile uint32_t triggerCount = 0;
unsigned long lastTriggerTime = 0;
volatile unsigned long lastHallMicros = 0;

volatile uint32_t drumTriggerCount = 0;
unsigned long lastDrumTriggerTime = 0;
volatile unsigned long lastDrumMicros = 0;

void IRAM_ATTR hallISR() {
  unsigned long now = micros();
  if (now - lastHallMicros < 300) return;
  lastHallMicros = now;
  triggerCount++;
  lastTriggerTime = millis();
}

void IRAM_ATTR drumISR() {
  unsigned long now = micros();
  if (now - lastDrumMicros < 300) return;
  lastDrumMicros = now;
  drumTriggerCount++;
  lastDrumTriggerTime = millis();
}

bool recording = false;
bool autoMode = false;
float autoStartRPM = 0;
float autoEndRPM = 0;

struct RunPoint { float rpm, hp, torque; };
std::vector<RunPoint> currentRun;

float runPeakHP = 0, runPeakHP_RPM = 0;
float runPeakTorque = 0, runPeakTorque_RPM = 0;

bool settingsPageActive = false;

Preferences preferences;

String sta_ssid = "";
String sta_password = "";
String currentCustomer = "";
String currentUnit = "";
String currentComments = "";

unsigned long recordingStartedMs = 0;

// Calibration task control
TaskHandle_t calibTask = NULL;
volatile float calibKnownWeight = 0.0f;
volatile bool calibRequest = false;

// MCP9600 console timing
unsigned long last_mcp_print = 0;
const unsigned long MCP_PRINT_EVERY_MS = 4000;

// ADDED: Software zero offset for tare
int32_t zeroOffset = 0;

void calibrationTask(void *pvParameters) {
  for (;;) {
    if (calibRequest) {
      calibRequest = false;

      ws.textAll("{\"type\":\"info\",\"msg\":\"Calibrating... hold weight still for 5 seconds (tare first!)\"}");

      float saved_factor = CALIBRATION_FACTOR;
      delay(2000);

      long raw_sum = 0;
      int total_samples = 0;
      int valid_count = 0;
      const int skip_first_n = 10;
      const unsigned long durationMs = 5000UL;
      const unsigned long sampleIntervalMs = 50;  // Faster with NAU7802

      unsigned long startTime = millis();
      unsigned long nextSampleTime = startTime;

      while (millis() - startTime < durationMs) {
        unsigned long now = millis();

        if (now >= nextSampleTime) {
          if (nau.available()) {
            long current = nau.read() - zeroOffset;
            total_samples++;

            if (labs(current) > 100000) {  // NAU raw values are larger
              if (valid_count >= skip_first_n) {
                raw_sum += current;
              }
              valid_count++;
            }
          }

          nextSampleTime += sampleIntervalMs;
        }
        delay(1);
      }

      int averaged_samples = valid_count - skip_first_n;
      if (averaged_samples < 5) {
        ws.textAll("{\"type\":\"info\",\"msg\":\"Not enough stable raw readings after skipping first 10 (" + 
                   String(averaged_samples) + "). Try again.\"}");
      } else {
        double avgRaw = (double)raw_sum / averaged_samples;

        // FIXED: Correct formula for NAU7802 when weight is entered in kg
        // factor = avgRaw / (kg * 1000) → so grams = raw * factor
        CALIBRATION_FACTOR = (calibKnownWeight * 1000.0f) / avgRaw;

        preferences.putFloat("cal_factor", CALIBRATION_FACTOR);
        broadcastSettings();

        ws.textAll("{\"type\":\"info\",\"msg\":\"Calibration done! New factor: " + 
                   String(CALIBRATION_FACTOR, 6) + " (stable avg raw " + String(avgRaw, 0) + 
                   " over " + String(averaged_samples) + " samples)\"}");

        // TEMPORARY DEBUG PRINTS (remove later if you want)
        Serial.print("Calib - avgRaw: "); Serial.println(avgRaw, 0);
        Serial.print("Known weight (kg): "); Serial.println(calibKnownWeight, 3);
        Serial.print("New CALIBRATION_FACTOR: "); Serial.println(CALIBRATION_FACTOR, 6);
      }
    }
    vTaskDelay(50 / portTICK_PERIOD_MS);
  }
}

float calculateDrumInertia() {
  float radius = DRUM_DIAM_METERS / 2.0f;
  if (DRUM_INERTIA_TYPE == "solid") {
    return 0.5f * DRUM_MASS * radius * radius;
  } 
  if (DRUM_INERTIA_TYPE == "hollow") {
    float inner_r = radius - (DRUM_WALL_THICKNESS_CM / 100.0f);
    if (inner_r <= 0) inner_r = 0.01f;
    return 0.5f * DRUM_MASS * (radius * radius + inner_r * inner_r);
  }
  return DRUM_INERTIA_CUSTOM;
}

void broadcastSettings() {
  String json = "{";
  json += "\"type\":\"settings\",";
  json += "\"maxRPM\":" + String(MAX_RPM, 0) + ",";
  json += "\"maxHP\":" + String(MAX_HP, 1) + ",";
  json += "\"maxTorque\":" + String(MAX_TORQUE, 1) + ",";
  json += "\"armCm\":" + String(LEVER_ARM_METERS * 100, 1) + ",";
  json += "\"calFactor\":" + String(CALIBRATION_FACTOR, 6) + ",";
  json += "\"pulsesPerRev\":" + String(PULSES_PER_REVOLUTION, 1) + ",";
  json += "\"rpmFilter\":" + String(RPM_FILTER_ALPHA, 2) + ",";
  json += "\"mode\":\"" + DYNO_MODE + "\",";
  json += "\"drumMass\":" + String(DRUM_MASS, 1) + ",";
  json += "\"drumCm\":" + String(DRUM_DIAM_METERS * 100, 1) + ",";
  json += "\"pulsesPerRevDrum\":" + String(PULSES_PER_REV_DRUM, 1) + ",";
  json += "\"drumInertiaType\":\"" + DRUM_INERTIA_TYPE + "\",";
  json += "\"drumWallCm\":" + String(DRUM_WALL_THICKNESS_CM, 1) + ",";
  json += "\"drumInertiaCustom\":" + String(DRUM_INERTIA_CUSTOM, 3) + ",";
  json += "\"rpmSource\":\"" + RPM_SOURCE + "\",";
  json += "\"virtGearRatio\":" + String(VIRTUAL_GEAR_RATIO, 2);
  json += "}";
  ws.textAll(json);
}

void notifyClients(const LiveData& data, bool isRecording) {
  String json = "{";
  json += "\"type\":\"data\",";
  json += "\"rpm\":" + String(data.rpm, 0) + ",";
  json += "\"hp\":" + String(data.hp, 2) + ",";
  json += "\"torque\":" + String(data.torque, 2) + ",";
  json += "\"recording\":" + String(isRecording ? "true" : "false") + ",";
  json += "\"peakHP\":" + String(data.peakHP, 2) + ",";
  json += "\"peakHP_RPM\":" + String(data.peakHP_RPM, 0) + ",";
  json += "\"peakTorque\":" + String(data.peakTorque, 2) + ",";
  json += "\"peakTorque_RPM\":" + String(data.peakTorque_RPM, 0) + ",";

  json += "\"mcps\":{";
  bool first = true;
  for (int i = 0; i < 8; i++) {
    if (mcp_oks[i]) {
      float hot = mcps[i].readThermocouple();
      float amb = mcps[i].readAmbient();
      if (!isnan(hot) && !isnan(amb)) {
        if (!first) json += ",";
        json += "\"" + String(i) + "\":{\"hot\":" + String(hot, 1) + ",\"amb\":" + String(amb, 1) + "}";
        first = false;
      }
    }
  }
  json += "}";
  json += "}";
  ws.textAll(json);
}

void sendRunComplete(float peakHP, float peakHP_RPM, float peakTorque, float peakTorque_RPM) {
  String json = "{";
  json += "\"type\":\"run_complete\",";
  json += "\"peakHP\":" + String(peakHP, 2) + ",";
  json += "\"peakHP_RPM\":" + String(peakHP_RPM, 0) + ",";
  json += "\"peakTorque\":" + String(peakTorque, 2) + ",";
  json += "\"peakTorque_RPM\":" + String(peakTorque_RPM, 0) + ",";

  json += "\"peakEGT\":" + String(peakEGT, 0) + ",";
  json += "\"finalEGT\":" + String(finalEGT, 0) + ",";
  json += "\"ambient\":" + String(runAmbient, 1);
  json += "}";
  ws.textAll(json);

  peakEGT = 0.0f;
  finalEGT = 0.0f;
  runAmbient = 0.0f;
}

void onWsEvent(AsyncWebSocket *server, AsyncWebSocketClient *client, AwsEventType type, void *arg, uint8_t *data, size_t len) {
  if (type == WS_EVT_DATA) {
    String msg = "";
    for (size_t i = 0; i < len; i++) msg += (char)data[i];
    msg.trim();

    if (msg == "TARE") {
      ws.textAll("{\"type\":\"info\",\"msg\":\"Taring... remove all load\"}");
      delay(1000);

      long sum = 0;
      int count = 0;
      for (int i = 0; i < 16; i++) {
        if (nau.available()) {
          sum += nau.read();
          count++;
          delay(5);
        }
      }
      if (count > 0) {
        zeroOffset = sum / count;
        ws.textAll("{\"type\":\"info\",\"msg\":\"Tare complete (zero offset set)\"}");
      } else {
        ws.textAll("{\"type\":\"info\",\"msg\":\"Tare failed - no data\"}");
      }
      return;
    }

    if (msg.startsWith("CALIBRATE:")) {
      float knownWeightKg = msg.substring(10).toFloat();
      if (knownWeightKg <= 0) {
        ws.textAll("{\"type\":\"info\",\"msg\":\"Enter valid weight > 0\"}");
        return;
      }

      calibKnownWeight = knownWeightKg;
      calibRequest = true;
      ws.textAll("{\"type\":\"info\",\"msg\":\"Calibration starting... Hold weight still for 5 seconds (tare first!)\"}");
      return;
    }

    if (msg.startsWith("SET_SINGLE:")) {
      String part = msg.substring(11);
      int eq = part.indexOf('=');
      if (eq == -1) return;

      String key = part.substring(0, eq);
      String valStr = part.substring(eq + 1);
      float val = valStr.toFloat();

      if (key == "maxRPM" && val >= 5000 && val <= 50000) MAX_RPM = val;
      else if (key == "maxHP" && val >= 5 && val <= 100) MAX_HP = val;
      else if (key == "maxTorque" && val >= 5 && val <= 100) MAX_TORQUE = val;
      else if (key == "armCm" && val >= 5 && val <= 100) {
        LEVER_ARM_METERS = val / 100.0f;
        preferences.putFloat("lever_arm", LEVER_ARM_METERS);
      }
      else if (key == "pulsesPerRev" && val >= 0.5f && val <= 10.0f) {
        PULSES_PER_REVOLUTION = val;
        preferences.putFloat("pulses_per_rev", val);
      }
      else if (key == "rpmFilter" && val >= 0.1f && val <= 0.9f) {
        RPM_FILTER_ALPHA = val;
        preferences.putFloat("rpm_filter", val);
      }
      else if (key == "dynoMode") {
        if (valStr == "brake" || valStr == "inertia") {
          DYNO_MODE = valStr;
          preferences.putString("dyno_mode", DYNO_MODE);
          if (valStr == "brake" && RPM_SOURCE != "spark") {
            RPM_SOURCE = "spark";
            preferences.putString("rpm_source", "spark");
            ws.textAll("{\"type\":\"info\",\"msg\":\"Brake mode selected — RPM Source forced to Engine Spark (Ignition Hall)\"}");
          }
        }
      }
      else if (key == "drumMass" && val >= 1.0f && val <= 300.0f) {
        DRUM_MASS = val;
        preferences.putFloat("drum_mass", DRUM_MASS);
      }
      else if (key == "drumCm" && val >= 10.0f && val <= 300.0f) {
        DRUM_DIAM_METERS = val / 100.0f;
        preferences.putFloat("drum_diam", DRUM_DIAM_METERS);
      }
      else if (key == "pulsesPerRevDrum" && val >= 0.5f && val <= 20.0f) {
        PULSES_PER_REV_DRUM = val;
        preferences.putFloat("pulses_drum", PULSES_PER_REV_DRUM);
      }
      else if (key == "drumInertiaType") {
        if (valStr == "solid" || valStr == "hollow" || valStr == "custom") {
          DRUM_INERTIA_TYPE = valStr;
          preferences.putString("drum_inertia_type", DRUM_INERTIA_TYPE);
        }
      }
      else if (key == "drumWallCm" && val >= 0.5f && val <= 20.0f) {
        DRUM_WALL_THICKNESS_CM = val;
        preferences.putFloat("drum_wall_cm", DRUM_WALL_THICKNESS_CM);
      }
      else if (key == "drumInertiaCustom" && val >= 0.01f && val <= 100.0f) {
        DRUM_INERTIA_CUSTOM = val;
        preferences.putFloat("drum_inertia_custom", DRUM_INERTIA_CUSTOM);
      }
      else if (key == "rpmSource") {
        if (valStr == "spark" || valStr == "drum" || valStr == "cvt") {
          if (DYNO_MODE == "brake" && valStr != "spark") {
            valStr = "spark";
            ws.textAll("{\"type\":\"info\",\"msg\":\"Brake mode only supports Engine Spark (Ignition Hall)\"}");
          }
          RPM_SOURCE = valStr;
          preferences.putString("rpm_source", RPM_SOURCE);
          if (valStr == "cvt") {
            VIRTUAL_GEAR_RATIO = 1.0f;
            preferences.putFloat("virt_gear_ratio", 1.0f);
            ws.textAll("{\"type\":\"info\",\"msg\":\"CVT mode active — using drum RPM directly (no fixed engine ratio)\"}");
          }
        }
      }
      else if (key == "virtGearRatio" && val >= 0.1f && val <= 200.0f) {
        if (RPM_SOURCE != "cvt" && DYNO_MODE != "brake") {
          VIRTUAL_GEAR_RATIO = val;
          preferences.putFloat("virt_gear_ratio", VIRTUAL_GEAR_RATIO);
        }
      }

      broadcastSettings();
      return;
    }

    if (msg.startsWith("WIFI_CLIENT:")) {
      int split = msg.indexOf(':', 12);
      if (split != -1) {
        sta_ssid = msg.substring(12, split);
        sta_password = msg.substring(split + 1);
        preferences.putString("sta_ssid", sta_ssid);
        preferences.putString("sta_password", sta_password);
        ws.textAll("{\"type\":\"info\",\"msg\":\"WiFi credentials saved. Rebooting to connect...\"}");
        delay(1000);
        ESP.restart();
      }
      return;
    }

    if (msg == "RESET_DEFAULT") {
      preferences.clear();
      ws.textAll("{\"type\":\"info\",\"msg\":\"Reset to default AP mode. Rebooting...\"}");
      delay(1000);
      ESP.restart();
      return;
    }

    if (msg.startsWith("AUTO_RUN:")) {
      int colon1 = msg.indexOf(':', 9);
      if (colon1 != -1) {
        autoStartRPM = msg.substring(9, colon1).toFloat();
        autoEndRPM = msg.substring(colon1 + 1).toFloat();
        if (autoStartRPM > 0 && autoEndRPM > autoStartRPM) {
          autoMode = true;
          ws.textAll("{\"type\":\"info\",\"msg\":\"Auto Run activated: Start ≥ " + String(autoStartRPM, 0) + " RPM, End ≥ " + String(autoEndRPM, 0) + " RPM\"}");
        } else {
          ws.textAll("{\"type\":\"info\",\"msg\":\"Invalid Auto Run RPM range\"}");
        }
      }
      return;
    }

    if (msg == "GET_SETTINGS") {
      broadcastSettings();
      return;
    }

    if (msg == "PAGE_SETTINGS") {
      settingsPageActive = true;
      return;
    }

    if (msg == "PAGE_MAIN") {
      settingsPageActive = false;
      return;
    }

    if (msg == "CANCEL_AUTO_RUN") {
      if (autoMode || recording) {
        autoMode = false;
        if (recording) {
          recording = false;
          recordingStartedMs = 0;
          sendRunComplete(runPeakHP, runPeakHP_RPM, runPeakTorque, runPeakTorque_RPM);
        }
        currentRun.clear();
        runPeakHP = 0; runPeakHP_RPM = 0;
        runPeakTorque = 0; runPeakTorque_RPM = 0;
        peakEGT = 0.0f;
        finalEGT = 0.0f;
        runAmbient = 0.0f;
        ws.textAll("{\"type\":\"info\",\"msg\":\"Run cancelled and reset\"}");
      }
      return;
    }

    if (msg == "MANUAL_START") {
      if (!recording) {
        recording = true;
        autoMode = false;
        currentRun.clear();
        runPeakHP = 0; runPeakHP_RPM = 0;
        runPeakTorque = 0; runPeakTorque_RPM = 0;
        peakEGT = 0.0f;
        finalEGT = 0.0f;
        runAmbient = 0.0f;
        recordingStartedMs = millis();
        ws.textAll("{\"type\":\"info\",\"msg\":\"Manual recording started\"}");
      } else {
        ws.textAll("{\"type\":\"info\",\"msg\":\"Recording already active\"}");
      }
      return;
    }

    if (msg == "MANUAL_STOP") {
      if (recording) {
        recording = false;
        autoMode = false;
        recordingStartedMs = 0;
        sendRunComplete(runPeakHP, runPeakHP_RPM, runPeakTorque, runPeakTorque_RPM);
        ws.textAll("{\"type\":\"info\",\"msg\":\"Manual recording stopped\"}");
      } else {
        ws.textAll("{\"type\":\"info\",\"msg\":\"No active recording to stop\"}");
      }
      return;
    }
  }
}

class CaptiveRequestHandler : public AsyncWebHandler {
public:
  bool canHandle(AsyncWebServerRequest *request) { return true; }
  void handleRequest(AsyncWebServerRequest *request) {
    request->redirect("/");
  }
};

void sensorTask(void * parameter) {
  for (;;) {
    if (!settingsPageActive) {
      static uint32_t lastCount = 0;
      static uint32_t lastDrumCount = 0;
      static unsigned long lastUpdate = 0;
      unsigned long now = millis();

      int updateInterval = 50;  // Can be faster now

      if (now - lastUpdate >= updateInterval) {
        float grams = 0.0f;
        if (nau.available()) {
          long raw = nau.read();
          grams = (float)(raw - zeroOffset) * CALIBRATION_FACTOR;
        }

        float kg = grams / 1000.0f;
        if (kg < 0) kg = 0;

        float force_N = kg * 9.81f;

        float engine_rpm_raw = 0;
        if (now - lastTriggerTime < 6000) {
          uint32_t pulses = triggerCount - lastCount;
          lastCount = triggerCount;
          float instant = (pulses > 0) ? (60000.0f / (now - lastUpdate)) * pulses / PULSES_PER_REVOLUTION : 0;
          rpmBuffer.pop_front();
          rpmBuffer.push_back(instant);
          float sum = 0; for (float v : rpmBuffer) sum += v;
          engine_rpm_raw = sum / BUFFER_SIZE;
        } else {
          rpmBuffer.assign(BUFFER_SIZE, 0.0f);
          engine_rpm_raw = 0;
        }

        static float engine_rpm_filtered = 0;
        engine_rpm_filtered = RPM_FILTER_ALPHA * engine_rpm_raw + (1.0f - RPM_FILTER_ALPHA) * engine_rpm_filtered;

        float drum_rpm_raw = 0;
        if (now - lastDrumTriggerTime < 6000) {
          uint32_t pulses = drumTriggerCount - lastDrumCount;
          lastDrumCount = drumTriggerCount;
          float instant = (pulses > 0) ? (60000.0f / (now - lastUpdate)) * pulses / PULSES_PER_REV_DRUM : 0;
          drumRpmBuffer.pop_front();
          drumRpmBuffer.push_back(instant);
          float sum = 0; for (float v : drumRpmBuffer) sum += v;
          drum_rpm_raw = sum / BUFFER_SIZE;
        } else {
          drumRpmBuffer.assign(BUFFER_SIZE, 0.0f);
          drum_rpm_raw = 0;
        }

        static float drum_rpm_filtered = 0;
        drum_rpm_filtered = RPM_FILTER_ALPHA * drum_rpm_raw + (1.0f - RPM_FILTER_ALPHA) * drum_rpm_filtered;
        float drum_rpm = drum_rpm_filtered;

        float engine_rpm;
        bool isCvtMode = (RPM_SOURCE == "cvt");
        if (isCvtMode || RPM_SOURCE == "drum") {
          if (isCvtMode) VIRTUAL_GEAR_RATIO = 1.0f;
          engine_rpm = drum_rpm * VIRTUAL_GEAR_RATIO;
          if (drum_rpm < 30) engine_rpm = 0;
          engine_rpm = constrain(engine_rpm, 0, MAX_RPM * 1.2);
        } else {
          engine_rpm = engine_rpm_filtered;
        }

        float used_rpm = drum_rpm;
        if (DYNO_MODE == "brake" && !isCvtMode) {
          used_rpm = (drum_rpm > 80) ? drum_rpm : engine_rpm;
        }
        if (used_rpm < 30) used_rpm = 0;

        float hp = 0.0f;
        float torque = 0.0f;

        if (DYNO_MODE == "brake") {
          float torque_drum = force_N * LEVER_ARM_METERS;
          torque = torque_drum;
          if (used_rpm > 50 && engine_rpm > 50) {
            hp = torque_drum * used_rpm / 7121.0f;
            torque = hp * 7121.0f / engine_rpm;
          } else {
            hp = 0.0f;
          }
        } else {  // inertia mode
          static std::deque<std::pair<float, float>> drumHistory;
          float t_s = static_cast<float>(now) / 1000.0f;
          drumHistory.push_back({t_s, drum_rpm});
          while (drumHistory.size() > 64) drumHistory.pop_front();

          if (drumHistory.size() < MIN_HIST_POINTS_INERTIA || drum_rpm < MIN_DRUM_RPM_INERTIA) {
            hp = 0;
            torque = 0;
          } else {
            size_t n = drumHistory.size();
            float sum_t = 0, sum_r = 0, sum_tr = 0, sum_t2 = 0;
            for (const auto& p : drumHistory) {
              sum_t += p.first;
              sum_r += p.second;
              sum_tr += p.first * p.second;
              sum_t2 += p.first * p.first;
            }
            float denom = n * sum_t2 - sum_t * sum_t;
            float slope_rpm_per_s = 0.0f;
            if (fabs(denom) > 1e-5f) {
              slope_rpm_per_s = (n * sum_tr - sum_t * sum_r) / denom;
            }

            static float slope_filtered = 0;
            slope_filtered = SLOPE_FILTER_NEW_ALPHA * slope_rpm_per_s + (1.0f - SLOPE_FILTER_NEW_ALPHA) * slope_filtered;

            if (slope_filtered < MIN_ACCEL_RPM_PER_S) {
              slope_filtered = 0;
            }

            float alpha_rad_s2 = (2.0f * 3.1415926535f / 60.0f) * slope_filtered;
            float I = calculateDrumInertia();
            float torque_drum = I * alpha_rad_s2;

            if (torque_drum < 0) torque_drum = 0;

            hp = torque_drum * drum_rpm / 7121.0f;
            torque = (engine_rpm > 50) ? (hp * 7121.0f / engine_rpm) : 0.0f;
          }
        }

        static std::deque<float> hp_buffer(9, 0.0f), torque_buffer(9, 0.0f);
        hp_buffer.pop_front(); hp_buffer.push_back(hp);
        torque_buffer.pop_front(); torque_buffer.push_back(torque);

        float hp_smooth = 0, torque_smooth = 0;
        for (auto v : hp_buffer) hp_smooth += v;
        for (auto v : torque_buffer) torque_smooth += v;
        hp_smooth /= hp_buffer.size();
        torque_smooth /= torque_buffer.size();

        const float AUTO_HYST = 180.0f;

        if (autoMode && !recording && engine_rpm >= (autoStartRPM + AUTO_HYST)) {
          recording = true;
          currentRun.clear();
          runPeakHP = 0; runPeakHP_RPM = 0;
          runPeakTorque = 0; runPeakTorque_RPM = 0;
          peakEGT = 0.0f;
          finalEGT = 0.0f;
          runAmbient = 0.0f;
          recordingStartedMs = millis();
          ws.textAll("{\"type\":\"info\",\"msg\":\"Auto Run started at " + String(engine_rpm, 0) + " RPM\"}");
        }

        if (autoMode && recording && engine_rpm >= (autoEndRPM - AUTO_HYST)) {
          recording = false;
          autoMode = false;
          recordingStartedMs = 0;
          sendRunComplete(runPeakHP, runPeakHP_RPM, runPeakTorque, runPeakTorque_RPM);
        }

        if (recording && (millis() - recordingStartedMs > 45000UL)) {
          recording = false;
          autoMode = false;
          ws.textAll("{\"type\":\"info\",\"msg\":\"SAFETY TIMEOUT - run stopped after 45s\"}");
          sendRunComplete(runPeakHP, runPeakHP_RPM, runPeakTorque, runPeakTorque_RPM);
          recordingStartedMs = 0;
        }

        if (recording) {
          if (hp_smooth > runPeakHP) { runPeakHP = hp_smooth; runPeakHP_RPM = engine_rpm; }
          if (torque_smooth > runPeakTorque) { runPeakTorque = torque_smooth; runPeakTorque_RPM = engine_rpm; }

          if (mcp_oks[7]) {
            float hot = mcps[7].readThermocouple();
            float amb = mcps[7].readAmbient();
            if (!isnan(hot) && !isnan(amb)) {
              if (hot > peakEGT) peakEGT = hot;
              finalEGT = hot;
              runAmbient = amb;
            }
          }

          static unsigned long lastRecordTime = 0;
          if (now - lastRecordTime >= 100) {
            currentRun.push_back({engine_rpm, hp_smooth, torque_smooth});
            lastRecordTime = now;
          }
        }

        current = {engine_rpm, hp_smooth, torque_smooth, runPeakHP, runPeakHP_RPM, runPeakTorque, runPeakTorque_RPM};

        notifyClients(current, recording);

        lastUpdate = now;
      }
    }

    vTaskDelay(1);
  }
}

void setup() {
  Serial.begin(115200);
  delay(100);

  Serial.println("Starting LittleFS...");
  if (!LittleFS.begin(true)) {
    Serial.println("LittleFS mount FAILED! Formatting and retrying...");
    LittleFS.format();
    if (!LittleFS.begin()) {
      Serial.println("LittleFS still failed after format - continuing without filesystem");
    } else {
      Serial.println("LittleFS formatted and mounted");
    }
  } else {
    Serial.println("LittleFS mounted successfully");
  }

  Wire.begin(8, 9);
  Wire.setClock(100000);

  Serial.println("\nScanning MCP9600 addresses 0x60–0x67...");
  active_mcps = 0;
  for (int i = 0; i < 8; i++) {
    uint8_t addr = MCP_ADDRS[i];
    mcp_oks[i] = false;
    if (mcps[i].begin(addr)) {
      mcps[i].setThermocoupleType(MCP9600_TYPE_K);
      mcps[i].setFilterCoefficient(3);
      mcps[i].setADCresolution(MCP9600_ADCRESOLUTION_14);
      mcps[i].setAmbientResolution(RES_ZERO_POINT_0625);
      mcp_oks[i] = true;
      active_mcps++;
      Serial.printf("FOUND → MCP #%d at 0x%02X (Type K, 14-bit, filter 3)\n", i+1, addr);
    }
  }
  if (active_mcps == 0) {
    Serial.println("No MCP9600 devices found — check wiring / pull-ups / address jumpers");
  } else {
    Serial.printf("Scan complete: %d device(s) active\n", active_mcps);
  }

  // ADDED: NAU7802 initialization
  if (!nau.begin()) {
    Serial.println("NAU7802 not found! Check wiring / address (0x2A)");
    while (1) delay(10);
  }
  Serial.println("NAU7802 detected!");

  nau.setGain(NAU7802_GAIN_128);
  nau.setRate(NAU7802_RATE_320SPS);  // Corrected method; start fast, drop to NAU7802_RATE_80SPS if noisy

  delay(500);
  long sum = 0;
  int count = 0;
  for (int i = 0; i < 16; i++) {
    if (nau.available()) {
      sum += nau.read();
      count++;
      delay(5);
    }
  }
  if (count > 0) {
    zeroOffset = sum / count;
    Serial.print("Initial zero offset: ");
    Serial.println(zeroOffset);
  }

  preferences.begin("dyno", false);
  CALIBRATION_FACTOR     = preferences.getFloat("cal_factor", 3993.567890);
  LEVER_ARM_METERS       = preferences.getFloat("lever_arm", 0.25);
  PULSES_PER_REVOLUTION  = preferences.getFloat("pulses_per_rev", 1.0f);
  RPM_FILTER_ALPHA       = preferences.getFloat("rpm_filter", 0.50f);
  sta_ssid               = preferences.getString("sta_ssid", "");
  sta_password           = preferences.getString("sta_password", "");
  DYNO_MODE              = preferences.getString("dyno_mode", "brake");
  DRUM_MASS              = preferences.getFloat("drum_mass", 10.0f);
  DRUM_DIAM_METERS       = preferences.getFloat("drum_diam", 0.3f);
  PULSES_PER_REV_DRUM    = preferences.getFloat("pulses_drum", 1.0f);
  DRUM_INERTIA_TYPE      = preferences.getString("drum_inertia_type", "solid");
  DRUM_WALL_THICKNESS_CM = preferences.getFloat("drum_wall_cm", 2.0f);
  DRUM_INERTIA_CUSTOM    = preferences.getFloat("drum_inertia_custom", 0.5f);
  RPM_SOURCE             = preferences.getString("rpm_source", "spark");
  VIRTUAL_GEAR_RATIO     = preferences.getFloat("virt_gear_ratio", 1.0f);

  if (sta_ssid.length() > 0) {
    WiFi.mode(WIFI_STA);
    WiFi.begin(sta_ssid.c_str(), sta_password.c_str());
    Serial.print("Connecting to WiFi: ");
    Serial.println(sta_ssid);

    unsigned long start = millis();
    while (WiFi.status() != WL_CONNECTED && millis() - start < 10000) delay(500);

    if (WiFi.status() == WL_CONNECTED) {
      Serial.println("\nConnected! IP: " + WiFi.localIP().toString());
      WiFi.setHostname("karlsdyno");
      if (MDNS.begin("karlsdyno")) {
        Serial.println("mDNS started: http://karlsdyno.local");
      }
    } else {
      Serial.println("\nFailed. Falling back to AP mode.");
      WiFi.mode(WIFI_AP);
      WiFi.softAP(ap_ssid, ap_password);
      Serial.println("AP started: " + String(ap_ssid) + " IP: " + WiFi.softAPIP().toString());
    }
  } else {
    WiFi.mode(WIFI_AP);
    WiFi.softAP(ap_ssid, ap_password);
    Serial.println("AP started: " + String(ap_ssid) + " IP: " + WiFi.softAPIP().toString());
  }

  dnsServer.start(53, "*", WiFi.softAPIP());
  server.addHandler(new CaptiveRequestHandler());

  server.on("/favicon.ico", HTTP_GET, [](AsyncWebServerRequest *request) {
    request->send(200, "text/plain", "");
  });

  server.serveStatic("/", LittleFS, "/").setDefaultFile("index.html");

  server.on("/settings", HTTP_GET, [](AsyncWebServerRequest *request) {
    if (LittleFS.exists("/settings.html")) {
      request->send(LittleFS, "/settings.html", "text/html");
    } else {
      request->send(404, "text/plain", "settings.html not found in LittleFS");
    }
  });

  ws.onEvent(onWsEvent);
  server.addHandler(&ws);

  server.begin();
  server.serveStatic("/logo1.png", LittleFS, "/logo1.png");

  pinMode(HALL_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(HALL_PIN), hallISR, FALLING);

  pinMode(DRUM_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(DRUM_PIN), drumISR, FALLING);

  rpmBuffer = std::deque<float>(BUFFER_SIZE, 0.0f);
  drumRpmBuffer = std::deque<float>(BUFFER_SIZE, 0.0f);

  broadcastSettings();

  xTaskCreatePinnedToCore(
    calibrationTask,
    "CalibTask",
    8192,
    NULL,
    2,
    &calibTask,
    1
  );

  xTaskCreatePinnedToCore(
    sensorTask,
    "SensorTask",
    8192,
    NULL,
    1,
    NULL,
    0
  );

  Serial.println("Setup complete - ready (with NAU7802)");
}

void loop() {
  dnsServer.processNextRequest();
  ws.cleanupClients();

  static unsigned long lastPrint = 0;
  if (millis() - lastPrint >= 5000) {
    lastPrint = millis();
    Serial.printf("Alive | Free heap: %d bytes | Uptime: %lu ms | Active MCPs: %d\n", ESP.getFreeHeap(), millis(), active_mcps);
  }

  if (millis() - last_mcp_print >= MCP_PRINT_EVERY_MS) {
    last_mcp_print = millis();
    bool any_ok = false;
    for (int i = 0; i < 8; i++) {
      if (mcp_oks[i]) {
        any_ok = true;
        float hot = mcps[i].readThermocouple();
        float amb = mcps[i].readAmbient();
        if (isnan(hot) || isnan(amb)) {
          Serial.printf("[MCP%d @ 0x%02X] READ ERROR ─── NaN returned\n", i+1, MCP_ADDRS[i]);
        } else {
          Serial.printf("[MCP%d @ 0x%02X] OK | Hot: %6.1f °C   Amb: %5.1f °C\n", i+1, MCP_ADDRS[i], hot, amb);
        }
      }
    }
    if (!any_ok) {
      Serial.println("[MCP] OFFLINE ─── No devices detected");
    }
  }

  delay(1);
}